
rm(list=ls())
library(dplyr)
library(Synth)

dat = read.csv("RAS.csv", header=TRUE)

dat$country = as.character(dat$country)

dat = dat[, c("country", "ctyid", "year", "polity", "gdppc", "pcturban", "rdi", "muslim", "pctmuslim", 
              "state", "social", "legislation")]


set.seed(12345)

##############################################################
# Create not.in function to make life easier
##############################################################
`%not.in%` <- function(x,y)!('%in%'(x,y))



##############################################################
# Delete Incomplete Countries as Preparation for Synthetic Control
##############################################################

num.missing <- function(x) {
  return (sum(is.na(x)))
}

dat.temp = dat
out = aggregate(x=dat.temp, by=list(country=dat.temp$country), FUN=num.missing)

has.missing = rowSums((out[,-1])) > 0
countries.without.missing = out$country[!has.missing]

dat = dat[dat$country %in% countries.without.missing,]




##############################################################
# synthetic control estimation
##############################################################

build.data <- function(dat.synth, predictors, dependent, treatment.identifier, controls.identifier, 
                       time, time.pretreat) {
  
  return(
    dataprep(
      foo = dat.synth,
      predictors = predictors,
      predictors.op = "mean",
      dependent = dependent,
      unit.variable = "ctyid",
      time.variable = "year",
      treatment.identifier = treatment.identifier,
      controls.identifier = controls.identifier,
      time.predictors.prior = time.pretreat,
      time.optimize.ssr = time.pretreat,
      unit.names.variable = "country",
      time.plot = time
    )
  )
}


draw.gap <- function(diff.mat, x.range, ylim, ylab, time.v, text.x, text.y, text) {
  plot(x.range, diff.mat[nrow(diff.mat),],
       type="l", ylim=ylim, 
       col=rgb(0,0,0,alpha=1), lty="solid",
       ylab = ylab,
       xlab ="Year",
       lwd=7,
       axes = FALSE,
       cex.lab = 1.5
  )
  axis(side=1, at=seq(time[1], time[length(time)], 2))
  axis(side=2, at=seq(ylim[1], ylim[2], 5))
  
  abline(v=time.v, lty="dotted")
  text(text.x, text.y, text, cex=1.4)
  abline(h=0, lty="dotted")
  
  for (i in 1:(nrow(diff.mat)-1)) {
    lines(x.range, diff.mat[i,], lwd=.1)  
  }
}


##############################################################
# defining global parameters
##############################################################
predictors.global = c("polity", "gdppc", "pcturban", "rdi", 
                      "state", "social", "legislation")


time = 1990:2014
time.pretreat = 1990:1997

# global parameters for graphic
width = 3200
height = 2400
res = 300

list.countries = unique(dat$country[dat$country %not.in% c("Indonesia")])
list.countries = c(list.countries, "Indonesia")


##############################################################
# social
##############################################################

dependent = "social"
predictors = unique(c(predictors.global, dependent))

excluded = c("China","France","Gambia",
             "Greece","Israel","Jamaica","Jordan",
             "Madagascar","Mauritania","Mongolia",
             "Norway","Poland","Senegal",
             "Tunisia","Turkey","USA")

list.treatment.identifier = list.countries[list.countries %not.in% excluded]


diff.mat = matrix(NA, nrow=length(list.treatment.identifier), ncol=length(time))

for (i in 1:length(list.treatment.identifier)) {
  treatment.identifier = list.treatment.identifier[i]
  controls.identifier = unique(dat$country[dat$country %not.in% c(treatment.identifier, "Indonesia")])  

  dataprep.out <- build.data(dat.synth = dat,
                             predictors = predictors, 
                             dependent = dependent, 
                             treatment.identifier, 
                             controls.identifier,
                             time,
                             time.pretreat)
  
  print(paste0(i," = ",treatment.identifier))
    
  synth.out <- synth(dataprep.out)
  Y0 = (dataprep.out$Y0%*%synth.out$solution.w)
  Y1 = dataprep.out$Y1plot
  diff.mat[i,] = Y1-Y0
}

row.names(diff.mat) <- list.treatment.identifier

#delete extreme countries with no reasonable synthetic control
diff.mat = diff.mat[abs(diff.mat[,1]) < 5,]

png(filename = "Figure A4.4. In-Space Placebo for Social Discrimination.png", width=width, height=height, res=res)
draw.gap(diff.mat, 
         x.range = time, 
         ylim = c(-10,10), 
         ylab = "Gap in Social Discrimination of Religious Minorities",
         time.v = 1998, 
         text.x = 1998, 
         text.y = -8, 
         text = "Soeharto's\nDownfall")
dev.off()


#p-value = 0.07
sum(abs(diff.mat[,25]) > abs(diff.mat[nrow(diff.mat),25]))/nrow(diff.mat)
